<?php

namespace Mnv\Admin\Controllers;

use Mnv\Core\ConfigManager;
use Mnv\Core\Design;
use Mnv\Http\Request;

/**
 * Class IndexAdmin
 * @package Mnv\Admin\Controllers
 */
class IndexAdmin
{
    /** @var mixed $backendController */
    private $backendController;

    /**
     * IndexAdmin constructor.
     *
     * @param Design $templateEngine
     * @throws \Exception
     */
    public function __construct(Design $templateEngine)
    {
        // Initialize language settings
        $adminLanguage = ConfigManager::getValue('admin_language');
        $this->initializeLanguage($templateEngine, $adminLanguage);

        // Set template configurations
        $this->configureTemplateEngine($templateEngine, $adminLanguage);

        // Load the requested module
        $moduleControllerName = $this->getRequestedModule();
        if ($moduleControllerName) {
            $this->initializeBackendController($moduleControllerName);
        } else {
            redirect('/admin/auth');
        }

//        print_r($this);
    }


    /**
     * Initialize language settings for the admin panel.
     *
     * @param  Design  $templateEngine
     * @param  string  $language
     *
     * @return void
     * @throws \Exception
     */
    private function initializeLanguage(Design $templateEngine, string $language): void
    {
        loadLanguage($language);
        $templateEngine->assign('adminLang', $language);
        $templateEngine->assign('jsLang', langJs('javascript'));

        try {
            $templateEngine->registerFilter(\Smarty::FILTER_PRE, 'replaceLanguageTags');
        } catch (\SmartyException $e) {
            throw new \Exception($e->getMessage());
        }
    }

    /**
     * Configure the template engine for the admin panel.
     *
     * @param Design $templateEngine
     * @param string $adminLanguage
     */
    private function configureTemplateEngine(Design $templateEngine, string $adminLanguage): void
    {
        $templateEngine->compile_check = \Smarty::COMPILECHECK_CACHEMISS;
        $templateEngine->setCompileDir($templateEngine->tmpPath() . 'compile');
        $templateEngine->setCompileId('admin-' . $adminLanguage);
        $templateEngine->setTemplateDir($templateEngine->defaultAdminTemplateDir());
        $templateEngine->debugging = false;
    }

    /**
     * Get the requested module name from the URL parameters.
     *
     * @return string|null
     */
    private function getRequestedModule(): ?string
    {
        $moduleName = preg_replace("/[^a-z\-]+/", "", request()->get('mod'));
        return !empty($moduleName) ? $this->findModule($moduleName) : null;
    }

    /**
     * Initialize the backend controller for the requested module.
     *
     * @param string $moduleControllerName
     * @throws \Exception
     */
    private function initializeBackendController(string $moduleControllerName): void
    {
        $moduleFilePath = GLOBAL_ROOT . "/admin/Controllers/" . $moduleControllerName . '.php';

        if (!file_exists($moduleFilePath)) {
            throw new \Exception("Module file not found: $moduleControllerName.php");
        }

        require_once $moduleFilePath;
        $controllerClass = '\\Mnv\\Admin\\Controllers\\' . $moduleControllerName;


        if (class_exists($controllerClass)) {
            $this->backendController = new $controllerClass();

        } else {
            throw new \Exception("Controller class not found: $controllerClass");
        }
    }

    /**
     * Render the requested module.
     *
     * @return string
     */
    public function fetch(): string
    {
        global $smarty;

        $content = $this->backendController->fetch();
        $controllerClass = get_class($this->backendController);

        if ($this->requiresAuthentication($controllerClass)) {
            $smarty->assign("content", false);
        } else {
            $smarty->assign("content", $content);
        }

        // Determine wrapper template  /** Создаем текущую обертку admin (index.tpl) */
        $wrapper = $smarty->getTemplateVars('wrapper') ?? 'index.tpl';

        /** Проверяем нам передали итоговую HTML или имя файла шаблона */
        if (!empty($wrapper)) {
            try {
                return $smarty->fetch($wrapper);
            } catch (\SmartyException | \Exception $e) {
                return $e->getMessage();
            }
        }

        return $content;
    }

    /**
     * Check if the controller requires authentication.
     *
     * @param string $controllerClass
     * @return bool
     */
    private function requiresAuthentication(string $controllerClass): bool
    {
        return ($controllerClass === 'AuthAdmin' || $this->getPermissionByController($controllerClass) === 'auth') || ($controllerClass === 'AuthAjaxAdmin' || $this->getPermissionByController($controllerClass) === 'ajaxauth');
    }


    /**
     * Get permission for the specified controller.
     *
     * @param $controller
     * @return false|mixed
     */
    public function getPermissionByController($controller)
    {
        $controllersPermissions = $this->getControllersPermissions();
        return $controllersPermissions[$controller] ?? false;
    }

    /**
     * Retrieve all controllers and their permissions.
     *
     * @return array
     */
    private function getControllersPermissions(): array
    {
        return include GLOBAL_ROOT . '/admin/config/modules.permissions.inc.php';
    }

    /**
     * Find the appropriate module for the given name.
     *
     * @param string $module
     * @return string
     */
    private function findModule(string $module): string
    {
        $modulesPermissions = $this->getControllersPermissions();

        foreach ($modulesPermissions as $model => $permission) {
            if (is_array($permission) && in_array($module, $permission, true)) {
                return $model;
            }

            if ($module === $permission) {
                return $model;
            }
        }

        return 'PageError';
    }

}