<?php

namespace Mnv\Core\Auth;

use Mnv\Core\Test\Log;
use Mnv\Core\UserInfo;
use Mnv\Core\Uploads\ImageSizes;

use Mnv\Core\Utilities\Cookie\Cookie;
use Mnv\Models\UserGroups;

use Mnv\Core\Utilities\GUID;
use Mnv\Core\Utilities\Cookie\Session;
use Mnv\Core\Utilities\Base64\Base64;

use Mnv\Core\Auth\Exceptions\UnknownIdException;
use Mnv\Core\Auth\Exceptions\UnknownUsernameException;
use Mnv\Core\Auth\Exceptions\ValidatePasswordException;

/**
 * Class AdminManager
 * @package Mnv\Core\Auth
 * @internal
 */
abstract class AdminManager
{
    const COOKIE_PREFIXES = [ Cookie::PREFIX_SECURE, Cookie::PREFIX_HOST ];
    const COOKIE_CONTENT_SEPARATOR = '~';

    const SESSION_FIELD_LOGGED_IN = 'admin.auth_logged_in';
    const SESSION_FIELD_USER_ID = 'admin.auth_user_id';
    const SESSION_FIELD_USERNAME = 'admin.auth_username';
    const SESSION_FIELD_STATUS = 'admin.auth_status';
    const SESSION_FIELD_ROLES = 'admin.auth_roles';
    const SESSION_FIELD_REMEMBERED = 'admin.auth_remembered';
    const SESSION_FIELD_LAST_RESYNC = 'admin.auth_last_resync';
    const SESSION_FIELD_FORCE_LOGOUT = 'admin.auth_force_logout';
    const SESSION_FIELD_BANNED = 'admin.auth_banned';
    const SESSION_FIELD_DEVICE = 'admin.auth_device';


    /** @var string */
    protected string $table = 'users';

    /** @var string  */
    protected string $primaryKey = 'userId';

    /** @var string  */
    protected string $columns = 'userId, password, verified, loginName, phone, status, groupId, force_logout';

    /** @var int|null $maxAttempts количество попыток */
    public ?int $maxAttempts = 10;

    /** @var int|null $blockDuration блокировка в секундах 1800 - 30 минут */
    public ?int $blockDuration = 1800;

    /** @var int|null $warningThreshold Предупреждение после 5 неправильных попыток */
    public ?int $warningThreshold = 5;

    /** @var int|null $retryDelay Задержка перед новой попыткой (в секундах) после 5 неудач */
    public ?int $retryDelay = 10;

    /** @var bool Включить защиту */
    public bool $blockHacking = false;

    /** @var string ID устройства */
    protected string $deviceId = '';


    /** AdminManager constructor. */
    protected function __construct()
    {
        $config = include GLOBAL_ROOT .'/includes/auth.config.inc.php';

        $this->blockHacking     = $config['blockHacking'];
        $this->maxAttempts      = $config['maxAttempts'];
        $this->blockDuration    = $config['blockDuration'];       // 30 минут
        $this->warningThreshold = $config['warningThreshold'];    // Предупреждение после 5 неправильных попыток
        $this->retryDelay       = $config['retryDelay'];          // Задержка перед новой попыткой (в секундах) после 5 неудач
    }

    /**
	 * Обновляет пароль указанного пользователя, устанавливая его на новый указанный пароль.
	 *
	 * @param int $userId ID идентификатор пользователя, пароль которого необходимо обновить
	 * @param string $newPassword новый пароль
	 * @throws UnknownIdException если пользователь с указанным идентификатором не найден
	 */
	protected function updatePasswordInternal(int $userId, string $newPassword)
    {
		$newPassword = \password_hash($newPassword, \PASSWORD_DEFAULT);
		$affected = connect('users')->where('userId',  $userId)->update(['password' => $newPassword]);
		if ($affected === 0) {
		    throw new UnknownIdException();
		}
	}

	/**
	 * Вызывается, когда пользователь успешно вошел в систему.
	 * Это может произойти при стандартном входе в систему, с помощью функции «запомнить меня» или из-за выдачи себя за другого человека администраторами.
	 *
	 * @param int $userId the ID of the userId
	 * @param string $loginName отображаемое имя (если есть) пользователя
	 * @param int $status статус пользователя как одна из констант из класса {@see Status}
	 * @param int $role роли пользователя
	 * @param int $forceLogout счетчик, отслеживающий принудительные выходы из системы, которые необходимо выполнить в текущем сеансе
	 * @param bool $remembered был ли пользователь запомнен
	 */
	protected function onLoginSuccessful(int $userId, string $loginName, int $status, int $role, int $forceLogout, bool $remembered = false)
    {
        // повторно сгенерировать идентификатор session, чтобы предотвратить атаки фиксации session (запрашивает запись cookie на клиенте)
        Session::regenerate(true);

		// сохраняем пользовательские данные в переменных session, поддерживаемых этой библиотекой
        $_SESSION[self::SESSION_FIELD_BANNED]       = 0;
        $_SESSION[self::SESSION_FIELD_LOGGED_IN]    = true;
        $_SESSION[self::SESSION_FIELD_USER_ID]      = $userId;
        $_SESSION[self::SESSION_FIELD_USERNAME]     = $loginName;
        $_SESSION[self::SESSION_FIELD_STATUS]       = $status;
        $_SESSION[self::SESSION_FIELD_ROLES]        = $role;
        $_SESSION[self::SESSION_FIELD_FORCE_LOGOUT] = $forceLogout;
        $_SESSION[self::SESSION_FIELD_REMEMBERED]   = $remembered;
        $_SESSION[self::SESSION_FIELD_LAST_RESYNC]  = \time();
	}

    /**
     * Возвращает данные пользователя для учетной записи с указанным именем входа (если есть)
     *
     * @param string $login the login to look for
     * @return array the user data (if an account was found unambiguously)
     * @throws UnknownUsernameException если не найден пользователь с указанным логином
     */
    protected function getUserDataByUsername(string $primaryKey, string $login): array
    {
        $user = connect($this->table)
            ->select($this->columns)
            ->where($primaryKey, $login)
            ->where('status',Status::NORMAL)
            ->in('userType', [UserGroups::DEVELOPER, UserGroups::ADMIN, UserGroups::MANAGER])
            ->get('array');

        // Обработка результата
        if (empty($user)) {
            // Пользователь не найден.
            $this->logFailedAttempt(UserInfo::get_ip(), "User not found : $login");
            throw new UnknownUsernameException();
        }

        $this->updateLoginUserIdAttempt($user['userId']);
        return $user;
    }

	/**
	 * Проверяет пароль
	 *
	 * @param string $password пароль для проверки
	 * @return string очищенный пароль
	 * @throws ValidatePasswordException если пароль был недействительным
	 */
	protected static function validatePassword(string $password): string
    {
		if (empty($password)) {
			throw new ValidatePasswordException();
		}

		$password = trim($password);
		if (strlen($password) < 6) {
			throw new ValidatePasswordException();
		}

		return $password;
	}


	/**
	 * Удаляет существующую директиву, которая удерживает пользователя в системе («запомни меня»).
	 *
	 * @param int $userId ID идентификатор пользователя, которого больше не следует оставлять в системе
	 * @param string|null $selector (optional) the selector which the deletion should be restricted to
	 */
	protected function deleteRememberDirectiveForUserById(int $userId, string $selector = null)
    {
		if (isset($selector)) connect()->where('selector', $selector);

		connect('users_remembered')->where('user', $userId)->delete();
	}

	/**
	 * Запускает принудительный выход из системы во всех сеансах, принадлежащих указанному пользователю.
	 *
	 * @param int $userId ID пользователя для выхода
	 */
	protected function forceLogoutForUserById(int $userId)
    {
		$this->deleteRememberDirectiveForUserById($userId);

        connect('users')->where('userId', $userId)->increment('force_logout', 1);
	}


    /**
     * Получение менеджера
     *
     * @param int|null $managerId
     * @return array|null
     */
    public function adminManager(?int $managerId): ?array
    {
        // Early return if managerId is not provided
        if (empty($managerId)) return null;

        // Fetch manager data
        $manager = connect('users AS u')
            ->usingJoin('user_group AS ug', 'groupId')
            ->leftJoin('user_images AS ui', 'ui.userId', '=', 'u.userId')
            ->leftJoin('files AS f', 'ui.fileId', '=', 'f.fileId')
            ->select('u.userId, u.fullName, u.firstName, u.middleName, u.lastName, u.loginName, u.userType, ug.groupId, ug.groupName, f.path, f.fileName')
            ->where('u.userId', $managerId)
            ->in('u.userType', [UserGroups::DEVELOPER, UserGroups::ADMIN, UserGroups::MANAGER])
            ->get('array');
//        print_r(connect()->getQuery().PHP_EOL);
        // Return null if no manager found
        if (empty($manager)) return null;

        $manager['image'] = !empty($manager['path']) ? ImageSizes::init()->get($manager, $manager) : null;

        // Fetch manager's permissions if groupId exists
        if (!empty($manager['groupId'])) {
            $manager['permissions'] = connect('user_group_privileges')->where('groupId', $manager['groupId'])->pluck('privilege', 'privilege');
        }

        return $manager;
    }


    /** **************** БЛОКИРОВКА ПОЛЬЗОВАТЕЛЯ ПРИ ПОТКИ ПОДБОРА ВХОДА *************** */

    protected function getUserAttempts()
    {
      return  connect('user_login_attempts')
          ->select('attempts, blocked_until, last_attempt')
          ->where('deviceId',  $this->deviceId)
          ->get('array');
    }

    /** Метод увеличения количества попыток */
    protected function incrementLoginAttempts(string $login = null)
    {
        $currentTime = date('Y-m-d H:i:s');

        // Prepare the data for the upsert
        $data = [
            'deviceId'      => $this->deviceId,
            'login'         => $login,
            'visitorIp'     => UserInfo::get_ip(),
            'attempts'      => 1,
            'last_attempt'  => $currentTime
        ];

        // Use upsert to insert or update
        $this->upsertLoginAttempt($data);
    }

    private function upsertLoginAttempt(array $data): void
    {
        connect('user_login_attempts')->upsert($data, [
            'attempts'      => 'attempts + 1',
            'login'         => $data['login'],
            'last_attempt'  => $data['last_attempt']
        ]);

    }

    /** Сбрасываем все неудачные попытки  */
    protected function resetLoginAttempts()
    {
        $this->updateUserAttempt([
            'attempts' => 0,
            'blocked_until' => NULL
        ]);
    }

    /** Обновляем данные о пользователе */
    protected function updateLoginUserIdAttempt($userId)
    {
        $this->updateUserAttempt([
            'userId' => $userId
        ]);
    }

    /** Блокировка пользователя */
    protected function blockUser()
    {
        $blockedUntil = date('Y-m-d H:i:s', time() + $this->blockDuration);
        $this->updateUserAttempt([
            'blocked_until' => $blockedUntil
        ]);
    }

    protected function tooManyBlockUser()
    {
        $blockedUntil = date('Y-m-d H:i:s', time() + $this->blockDuration);
        $this->updateUserAttempt([
            'attempts' => $this->maxAttempts,
            'blocked_until' => $blockedUntil
        ]);
    }

    private function updateUserAttempt(array $data)
    {
        connect('user_login_attempts')
            ->where('deviceId', $this->deviceId)
            ->orWhere('visitorIp', UserInfo::get_ip())
            ->update($data);
    }

    protected function logFailedAttempt($ip, $reason)
    {
        $log = new Log('login_attempts.log');

        $log->write("Failed login attempt from IP: $ip, Reason: $reason");
    }


    /**
     * Получаем идентификатор устройства
     * @return mixed|string
     */
    protected function createDeviceId()
    {
        // Проверяем, есть ли уже session с идентификатором устройства
        if (isset($_SESSION[self::SESSION_FIELD_DEVICE])) {
            return $_SESSION[self::SESSION_FIELD_DEVICE];
        }

        // Генерируем новый уникальный идентификатор
        $deviceId = GUID::Format(GUID::Create(), false, '-');
        $_SESSION[self::SESSION_FIELD_DEVICE] = $deviceId;

        return $deviceId;
    }

    /**
     * Создает случайную строку с заданной максимальной длиной
     *
     * С параметром по умолчанию вывод должен содержать как минимум столько же случайности, сколько UUID.
     *
     * @param int $maxLength
     * @return string
     */
    public static function createRandomString(int $maxLength = 24): string
    {
        // вычислить, сколько байтов случайности нам нужно для указанной длины строки
        $bytes = \floor($maxLength / 4) * 3;

        // получить случайные данные
        $data = \openssl_random_pseudo_bytes($bytes);

        // вернуть результат в кодировке Base64
        return Base64::encodeUrlSafe($data);
    }

}