<?php

namespace Mnv\Core;

use Mnv\Core\Database\Throwable\DatabaseException;
use Mnv\Core\Uploads\ImageSizes;
use Mnv\Models\Exceptions\NoContentException;
use Mnv\Models\Exceptions\NotFoundException;
use Mnv\Models\Interfaces\IModel;

/**
 * Class Model
 * @package Mnv\Core
 */
abstract class Model implements IModel
{
    /**
     * Таблица, связанная с моделью.
     *
     * @var string
     */
    protected string $table = '';

    /**
     * Таблица файлов, связанная с моделью.
     *
     * @var string
     */
    protected string $table_image = '';

    /**
     * Первичный ключ для модели.
     *
     * @var string
     */
    protected string $primaryKey = 'id';

    /**
     * @var string
     */
    protected string $orderBy = '';

    /**
     * @var string
     */
    protected string $columns = '*';

    /**
     * Указывает, существует ли модель.
     *
     * @var bool
     */
    public bool $exists = false;

    /**
     * Идентификатор модели
     *
     * @var int|null
     */
    public ?int $id;

    /**
     * Массив модели
     *
     * @var array|null
     */
    public ?array $data = [];

    /**
     * @var array
     */
    public array $filter = [];

    /**
     * @var string|null
     */
    public ?string $status = '';


    public $favorites = '';

    /**
     * Кол-во
     * @var int
     */
    public int $total = 0;

    /**
     * @var array|null
     */
    public ?array $result = [];

    /**
     * @var string
     */
    public string $sortTableId = '';

    /**
     * @var string|null
     */
    public ?string $sortBy;

    /**
     * @var string|null
     */
    public ?string $sortOrder;

    /**
     * Установить таблицу, связанную с моделью.
     *
     * @param string $table
     * @return $this
     */
    public function setTable(string $table): Model
    {
        $this->table = $table;

        return $this;
    }

    /**
     * Получить таблицу, связанную с моделью.
     *
     * @return string
     */
    public function getTable(): string
    {
        return $this->table ?? strtolower(class_basename($this));
    }

    /**
     * Set and get columns for selection
     *
     * @param $columns
     */
    public function setColumn($columns): void
    {
        if (!empty($columns)) {
            if (is_array($columns)) {
                if (isset($columns)) {
                    unset($columns[array_search('image', $columns)]);
                }
                $this->columns = implode(", ", $columns);
            } else {
                $this->columns = $columns;
            }
        } else {
            $this->columns = '*';
        }
    }

    /**
     * @return string
     */
    public function getColumn(): string
    {
        return $this->columns;
    }

    /**
     * Set and get orderBy clause
     * @param $orderBy
     */
    public function setOrderBy($orderBy): void
    {
        $this->orderBy = $orderBy;
    }

    /**
     * @return string
     */
    public function getOrderBy(): string
    {
        return $this->orderBy;
    }


    /**
     * Применить сортировку на основе таблицы сортировки
     */
    protected function setSorting(): void
    {
        $sortable = SortTable::init()->getSort($this->sortTableId);
        if (!empty($sortable)) {
            $this->sortBy = $sortable[$this->sortTableId]['last_sort_by'] ?? null;
            $this->sortOrder = $sortable[$this->sortTableId]['last_sort_order'] ?? null;
            $this->orderBy = "{$this->sortBy} " . strtoupper($this->sortOrder);
        }

        if (!empty($this->orderBy)) {
            connect($this->table)->orderBy($this->orderBy);
        }
    }

    /**
     * Fetch all records with pagination
     * Получить все элементы, связанные с моделью.
     *
     * @param int $limit
     * @param int $page
     * @return mixed
     */
    public function all(int $limit, int $page)
    {
        $this->setSorting();

        $results = connect($this->table)->select($this->columns)->pagination($limit, $page)->getAll('array');
//        print_r(connect()->getQuery().PHP_EOL);

        // Пропускаем обработку, если данные пусты
        if ($results && !empty($this->table_image)) {
            $results = $this->prepareImages($results);
        }

//        print_r($results);
        return $results;
    }

    protected function prepareImages($results): array
    {
        $imageSizes = ImageSizes::init();

        // Оптимизированное получение первичных ключей
        $primaryKeys = array_unique(array_column($results, $this->primaryKey));

        // Если нет первичных ключей, возвращаем исходные результаты
        if (!$primaryKeys) {
            return $results;
        }

        // Получаем все изображения одним запросом с помощью IN
        $images = connect($this->table_image)->usingJoin('files', 'fileId')->where('type', 'general')->in($this->primaryKey, $primaryKeys)->getAll('array');
//        print_r(connect()->getQuery().PHP_EOL);
        if ($images) {
            // Создаем карту изображений по первичным ключам для быстрого доступа
            $imageMap = [];
            foreach ($images as $item) {
                $imageMap[$item[$this->primaryKey]] = $imageSizes->get($item, $item);
            }

            // Ассоциируем изображения с результатами, используя одну итерацию
            foreach ($results as &$result) {
                $primaryKey = $result[$this->primaryKey];
                if (isset($imageMap[$primaryKey])) {
                    $result['image'] = $imageMap[$primaryKey];
                }
            }
        }

        return $results;
    }


    /**
     * Count total records
     * @return void
     */
    public function total(): void
    {
        $this->total = (int)connect($this->table)->count('*', 'count')->getValue();
    }

    /**
     * Fetch a specific record by primary key
     * @return array|null
     */
    public function get(): ?array
    {
        return connect($this->table)->where($this->primaryKey,  $this->id)->get('array');
    }

    /**
     * Edit a record by ID
     * @return $this
     */
    public function edit(): Model
    {
       !empty($this->id) && $this->data = $this->get();

        return $this;
    }

    /**
     * Insert a new record
     * @param array $data
     * @return bool|int|string|void|null
     */
    public function insert(array $data)
    {
        return connect($this->table)->insert($data);
    }

    /**
     * Update an existing record
     * @param array $data
     * @return bool
     */
    public function update(array $data): bool
    {
        return !empty($this->id) && connect($this->table)->where($this->primaryKey, $this->id)->update($data);
    }


    /**
     * Toggles the status of the current record between 'V' (Visible) and 'H' (Hidden).
     *
     * @return bool True if status was successfully toggled.
     *
     * @throws NotFoundException if `id` is missing or invalid.
     * @throws NoContentException if no record is found for the given `id`.
     */
    public function status(): bool
    {
        // Ensure that `id` is provided; throw exception if it's missing
        if (empty($this->id)) {
            throw new NotFoundException("Not Found: ID is required.");
        }

        // Retrieve the current `status` from the database based on `id`
        $currentRecord = connect($this->table)->select('status')->where($this->primaryKey, $this->id)->get('array');

        // Throw exception if no record is found for the given `id`
        if (!$currentRecord) {
            throw new NoContentException("No Content: Record not found for the given ID.");
        }
        // Toggle status between 'V' (Visible) and 'H' (Hidden)
        $this->status = ($currentRecord['status'] === 'V') ? 'H' : 'V';
        connect($this->table)->where($this->primaryKey, $this->id)->update(['status' => $this->status]);

        return true;
    }


    /**
     * Delete a record
     * @return bool
     * @throws NotFoundException
     * @throws NoContentException
     * @throws DatabaseException
     */
    public function remove(): bool
    {
        if (empty($this->id)) {
            throw new NotFoundException("Not Found: ID is required.");
        }

        $currentRecord = connect($this->table)->where($this->primaryKey, $this->id)->get('array');

        // Throw exception if no record is found for the given `id`
        if (!$currentRecord) {
            throw new NoContentException("No Content: Record not found for the given ID.");
        }

        $deleted = connect($this->table)->where($this->primaryKey, $this->id)->delete();
        if (!$deleted) {
            throw new DatabaseException("Error: Failed to delete record with ID {$this->id}.");
        }

        // Delete from image table if it exists
        if (!empty($this->table_image)) {
            connect($this->table_image)->where($this->primaryKey, $this->id)->delete();
        }


        return true;
    }


    /**
     * Approve record
     *
     * @return bool
     * @throws NotFoundException
     * @throws NoContentException
     */
    public function approve(): bool
    {
        if (empty($this->id)) {
            throw new NotFoundException("Not Found: ID is required.");
        }

        $currentRecord = connect($this->table)->where($this->primaryKey, $this->id)->get('array');

        // Throw exception if no record is found for the given `id`
        if (!$currentRecord) {
            throw new NoContentException("No Content: Record not found for the given ID.");
        }

        if (connect($this->table)->where($this->primaryKey, $this->id)->update(['status' => 'V'])) {
            return true;
        }

        return false;
    }

    /**
     * Approve record
     *
     * @return bool
     * @throws NotFoundException
     * @throws NoContentException
     */
    public function favorites(): bool
    {
        if (empty($this->id)) {
            throw new NotFoundException("Not Found: ID is required.");
        }

        $currentRecord = connect($this->table)->where($this->primaryKey, $this->id)->get('array');

        // Throw exception if no record is found for the given `id`
        if (!$currentRecord) {
            throw new NoContentException("No Content: Record not found for the given ID.");
        }

        // Toggle 'isFeatured' between 1 (true) and 0 (false)
        $this->favorites = ($currentRecord['isFeatured'] == 1) ? 0 : 1;
        connect($this->table)->where($this->primaryKey, $this->id)->update(['isFeatured' => $this->favorites]);

        return true;
    }

    /**
     * Get the maximum value of a field
     * @param  string  $field
     * @param  string|null  $name
     *
     * @return mixed|null
     */
    protected function getMaxValue(string $field, ?string $name = null)
    {
        return connect($this->table)->max($field, $name)->getValue();
    }

    /**
     * Fetch user information (creator and modifier)
     * получение информации о пользователях, создавших и изменивших контент
     */
    public function gettingInfoAboutUsers(): Model
    {
        if (isset($this->data['addedBy'], $this->data['modifiedBy'])) {
            $managers = connect('users')->select('userId, loginName')->in('userId', [$this->data['addedBy'], $this->data['modifiedBy']])->keyBy( 'userId');
            $this->data['addedBy']    = !empty($managers[$this->data['addedBy']]) ? $managers[$this->data['addedBy']] : unknownUser();
            $this->data['modifiedBy'] = !empty($managers[$this->data['modifiedBy']]) ? $managers[$this->data['modifiedBy']] : unknownUser();
            $this->data['addedOn']    = adjustTime($this->data['addedOn'], false, 'd.m.Y H:i');
            $this->data['modifiedOn'] = adjustTime($this->data['modifiedOn'], false, 'd.m.Y H:i');
            if (isset($this->data['publishedOn']))
                $this->data['publishedOn'] = adjustTime($this->data['publishedOn']);
        }

        return $this;
    }



    /** IMAGE/FILE HANDLING */


    /**
     * Retrieve all related images
     */
    public function getFiles(): self
    {
        if (!empty($this->table_image)) {
            $images = connect($this->table_image)->usingJoin('files', 'fileId')->where($this->primaryKey, $this->id)->orderBy('orderBy ASC')->getAll('array');
            if ($images) {
                $this->processImages($images);
            }
        }

        return $this;
    }

    private function processImages(array $images): void
    {
        $imageSizes = ImageSizes::init();
        foreach ($images as $item) {
            $this->assignImageData($item, $item, $imageSizes);
        }
    }

    protected function getFileInfo($fileId): ?array
    {
        static $fileCache = [];
        if (!isset($fileCache[$fileId])) {
            $fileCache[$fileId] = connect('files')->where('fileId', $fileId)->get('array');
        }

        return $fileCache[$fileId];
    }


    private function assignImageData(array $item, array $file, ImageSizes $imageSizes): void
    {
        $imageData = $imageSizes->get($item, $file);

        switch ($item['type']) {
            case 'gallery':
                $this->data['gallery'][] = $imageData;
                break;
            case 'docs':
                $this->data['docs'][] = $imageData;
                break;
            case 'general':
            default:
                $this->data['image'] = $imageData;
                break;
        }
    }


    /**
     * Delete an image by ID
     *
     * @param $imageId
     * @return bool
     */
    public function deleteFile($imageId): bool
    {
        if (connect($this->table_image)->where('imageId', $imageId)->delete())
            return true;

        return false;
    }

    /**
     * Update image information
     *
     * @param $imageId
     * @param $picture
     * @return bool
     */
    public function editPicture($imageId, $picture): bool
    {
        $updateParams = array_intersect_key($picture, array_flip(['title', 'alias', 'description', 'link', 'position']));

        if (connect($this->table_image)->where('imageId', $imageId)->update($updateParams)) {
            $this->result = connect($this->table_image)->where('imageId', $imageId)->get('array');
            return true;
        }

        return false;
    }

    /**
     * Sort images by order
     *
     * @param array $imageIds
     * @return bool
     */
    public function sortingPictures(array $imageIds): bool
    {
        if (empty($imageIds)) {
            return false;
        }

        // Подготовьте данные для пакетного обновления
        $updates = [];
        foreach ($imageIds as $i => $imageId) {
            $updates[] = ['imageId' => $imageId, 'orderBy' => $i + 1];
        }

        // Предполагая, что на уровне абстракции вашей базы данных существует метод массового обновления
        foreach ($updates as $update) {
            connect($this->table_image)->where('imageId', $update['imageId'])->update(['orderBy' => $update['orderBy']]);
        }

        return true;
    }


    /**
     * Add files (images/documents)
     *
     * @param int $modelId
     * @param array $files
     * @param string $type
     * @return bool
     */
    public function addFiles(int $modelId, array $files, string $type = 'general'): bool
    {

        // Проверяем, есть ли файлы для данного типа
        if (empty($files[$type])) {
            return false;
        }

        // Получаем текущий максимальный orderBy для заданного типа и модели
        $idx = connect($this->table_image)->where($this->primaryKey, $modelId)->where('type', $type)->max('orderBy')->getValue() ?? 0;

        if ($type == 'general') {
            if ($image = connect($this->table_image)->select('imageId')->where('type', 'general')->where($this->primaryKey, $modelId)->get('array')) {
                connect($this->table_image)->where('imageId', $image['imageId'])->where($this->primaryKey, $modelId)->delete();
            }
        }

        // Подготовка массива для массового обновления/вставки
        $uploads = [];

        foreach ($files[$type] as $file) {
            // Увеличиваем порядок
            $idx++;

            // Создаем массив для загрузки файла
            $fileUpload = array_merge([
                $this->primaryKey   => $modelId,
                'type'              => $type,
                'orderBy'           => $idx,
            ], array_intersect_key($file, array_flip(['fileId', 'title', 'alias', 'description', 'link', 'position'])));


            // Проверяем наличие imageId для обновления или добавления новой записи
            if (!empty($file['imageId'])) {
                connect($this->table_image)->where('imageId', $file['imageId'])->update($fileUpload);
            } else  {
                $uploads[] = $fileUpload;
            }
        }

        // Массовая вставка новых записей (если есть)
        if (!empty($uploads)) {
            connect($this->table_image)->insert($uploads);
        }

        return true;
    }

}