<?php

namespace Mnv\Core;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\ConnectException;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\Exception\RequestException;

use GuzzleHttp\HandlerStack;
use GuzzleHttp\Middleware;
use Mnv\Core\Test\Log;
use Psr\Http\Message\RequestInterface;
use Throwable;

class Oktell
{
    /** @var Client  */
    private Client $httpClient;

    /** @var Log  */
    private Log $logger;

    /** @var string  */
    private string $endpoint;


    /** @var bool  */
    private bool $verify;

    /** @var bool */
    private bool $debug;

    /** @var int  */
    private int $timeout;

    /** Константы для дефолтных значений */
    private const DEFAULT_TIMEOUT = 60;
    private const DEFAULT_CONNECT_TIMEOUT = 30;

    public function __construct(string $endpoint, bool $verify = false, bool $debug = false, int $timeout = self::DEFAULT_TIMEOUT)
    {
        $this->logger   = new Log('oktell.log');
        $this->endpoint = $endpoint;
        $this->verify   = $verify;
        $this->debug    = $debug;
        $this->timeout  = $timeout;

        $stack = HandlerStack::create();

        // Перехват и логирование запросов
        $stack->push(Middleware::tap(function (RequestInterface $request) {
            if ($this->debug) {
                $this->logger->write('Запрос :: ' . $request->getMethod() . ' ' . (string) $request->getUri());
                $headers = [];
                foreach ($request->getHeaders() as $name => $values) {
                    $headers[] =  $name . ': ' . implode(', ', $values);
                }

                $this->logger->write('Заголовки :: ' . implode("\n", $headers));
                $this->logger->write('Тело :: ' .  $request->getBody() );
            }

        }));

        // Инициализация Guzzle-клиента
        $this->httpClient = new Client([
            'verify'            => $this->verify,
            'debug'             => $this->debug,
            'http_errors'       => false,
            'timeout'           => $this->timeout,
            'connect_timeout'   => self::DEFAULT_CONNECT_TIMEOUT,
            'handler' => $stack
        ]);
    }

    public function send(array $data): array
    {
        if ($this->debug) {
            $this->logger->write('Отправка данных :: ' . json_encode([
                    'endpoint' => $this->endpoint,
                    'data' => $data
                ], JSON_UNESCAPED_UNICODE));
        }

        try {
            $response = $this->httpClient->post($this->endpoint, $this->getRequestOptions($data));
            if ($this->debug) {
                $this->logger->write('Успешный ответ от сервера :: ' . json_encode([
                        'statusCode' => $response->getStatusCode(),
                        'responseBody' => $response->getBody()->getContents()
                    ], JSON_UNESCAPED_UNICODE));
            }

            return $this->handleResponse(
                $response->getStatusCode(),
                $response->getBody()->getContents()
            );

        } catch (ConnectException $e) {
            if ($this->debug) {
                $this->logger->write('Ошибка соединения :: ' . json_encode([
                        'message' => $e->getMessage()
                    ], JSON_UNESCAPED_UNICODE));
            }
            return $this->handleError($e);

        } catch (RequestException $e) {
            if ($this->debug) {
                $request = $e->getRequest();
                $this->logger->write('Запрос :: ' . $request->getMethod() . ' ' . (string) $request->getUri());
                $this->logger->write('Заголовки :: ' . json_encode($request->getHeaders(), JSON_PRETTY_PRINT));
                $this->logger->write('Тело :: ' .  $request->getBody() );

                $this->logger->write('Guzzle error :: ' . json_encode([
                        'message' => $e->getMessage(),
                        'request' => $e->getRequest() ? $e->getRequest()->getBody() : 'N/A'
                    ], JSON_UNESCAPED_UNICODE));
            }

            return $this->handleError($e);

        } catch (Throwable $e) {
            if ($this->debug) {
                $this->logger->write('Ошибка при отправке данных :: ' . json_encode([
                        'message' => $e->getMessage(),
                        'trace' => $e->getTraceAsString()
                    ], JSON_UNESCAPED_UNICODE));
            }

            return $this->handleError($e);
        }
    }

    private function getRequestOptions(array $data): array
    {
        return [
            'headers' => [
                'Content-Type' => 'application/json',
            ],
            'json' => $data,
        ];
    }

    private function handleResponse(int $httpCode, string $responseBody): array
    {
        if ($httpCode === 204) {
            return [
                'success' => true,
                'response' => null,
                'httpCode' => $httpCode,
            ];
        }

        $responseData = json_decode($responseBody, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            return [
                'success' => $httpCode === 200,
                'response' => $responseBody,
                'error' => 'Ошибка декодирования JSON: ' . json_last_error_msg(),
            ];
        }

        return [
            'success' => $httpCode === 200,
            'response' => $responseData,
            'httpCode' => $httpCode,
        ];
    }

    private function handleError(Throwable $e): array
    {
        return [
            'success' => false,
            'error' => $e->getMessage(),
        ];
    }


}