<?php

namespace Mnv\Core\Test;

use Mnv\Core\DB;
use Mnv\Core\Singleton\SingletonTrait;

/**
 * Class Logger
 * @package Mnv\App\Test
 */
class Logger
{
    use SingletonTrait;

    private array $logs = [];
    private const MAX_LOGS = 100; // Max logs to keep in memory before saving

    /**
     * Add a message to the log.
     *
     * @param string|null $message The log message.
     * @param mixed $managerId The ID of the manager associated with the log.
     * @param string $type The type of log (log, info, error).
     * @return $this
     */
    public function log(?string $message, $managerId, string $type = "log"): self
    {
        $validTypes = ['log', 'info', 'error'];

        // Validate the log type
        if (!in_array($type, $validTypes, true)) {
            $type = 'log'; // Default type
        }

        // Append the log entry
        $this->logs[] = [
            'message'   => $message,
            'type'      => $type,
            'managerId' => $managerId,
            'date_log'  => gmdate('Y-m-d H:i:s'),
            'user_ip'   => '', // Optional: Can be populated later
        ];

        // Save logs if the maximum count is reached
        if (count($this->logs) >= self::MAX_LOGS) {
            $this->save();
        }

        return $this;
    }

    /**
     * Log an error message.
     *
     * @param string|null $message The error message.
     * @param mixed $managerId The ID of the manager associated with the error.
     * @param bool $save Whether to save the log.
     * @return $this
     */
    public function error(?string $message, $managerId, bool $save = true): self
    {
        return $save ? $this->log($message, $managerId, "error") : $this;
    }

    /**
     * Log an informational message.
     *
     * @param string|null $message The info message.
     * @param mixed $managerId The ID of the manager associated with the info.
     * @param bool $save Whether to save the log.
     * @return $this
     */
    public function info(?string $message, $managerId, bool $save = true): self
    {
        return $save ? $this->log($message, $managerId, "info") : $this;
    }

    /**
     * Save logs to the database.
     *
     * @return void
     */
    public function save(): void
    {
        if (empty($this->logs)) {
            return; // Early exit if there are no logs to save
        }

        // Prepare log entries for database insertion
        $dump = [];
        foreach ($this->logs as $logEntry) {
            $dump[] = [
                'userId'   => $logEntry['managerId'],
                'message'  => $logEntry['message'],
                'type_log' => $logEntry['type'],
                'date_log' => $logEntry['date_log'],
                'user_ip'  => $logEntry['user_ip'], // Consider populating with real IP if necessary
            ];
        }

        // Perform batch insertion with a transaction (if supported by your DB)
        connect()->transaction();
        try {
            connect('log')->insert($dump);
            connect()->commit();
        } catch (\Exception $e) {
            connect()->rollBack();
            // Handle logging or reporting the error appropriately
        }

        // Clear logs after saving to avoid duplicate entries
        $this->logs = [];
    }
}


