<?php

namespace Mnv\Core\Uploads;

use Mnv\Core\ConfigManager;
use Mnv\Core\Utilities\GUID;
use Symfony\Component\HttpFoundation\File\UploadedFile as SymfonyUploadedFile;

use Intervention\Image\Image;
use Intervention\Image\ImageManager;


/**
 * Class ImageGenerator
 * @package Mnv\Core\Uploads
 */
class ImageGenerator extends AbstractGenerator implements IGenerator
{

    /** @var Image */
    private Image $image;

    /** @var ImageManager  */
    private ImageManager $manager;

    public function __construct(string $realPath, string $path, ?string $uuid, int $managerId)
    {
        parent::__construct($realPath,  $path, $uuid, $managerId);
        $this->setDriver();
    }

    /**
     * @param SymfonyUploadedFile $file
     *
     * @return ImageGenerator
     */
    public function init(SymfonyUploadedFile $file): ImageGenerator
    {
        $this->file = $file;
        $this->generatorExtension($file->getClientOriginalExtension());
        /** новое название файла uuid */
        $this->generateFileName($file->getClientOriginalName());
        /** сбор опций обработки `watermark` */
        $this->getFontOptions();

        /** достаем временный файл для обработки */
        $this->image = $this->manager->make($this->file->getPathname());

        return $this;
    }

    /**
     * @return bool|int|string|null
     */
    public function save()
    {
        $images = [
            'name'      => "{$this->fileName}.{$this->extension}",
            'size'      => $this->file->getSize() ?? null,
            'mimeType'  => ($this->extension == 'webp') ? 'image/webp' : $this->file->getClientMimeType(),
        ];


        $this->resizeImage($this->image->getWidth(), $this->image->getHeight(),$this->realPath . $this->fileName . '.' . $this->extension);

        foreach ($this->sizes as $size) {
            $sizes = $this->autoSize(ConfigManager::getValue("{$size}_thumb"));
            $this->resizeImage($sizes[0], $sizes[1], $this->realPath  . $size . '/' . $this->fileName . '.' . $this->extension);
        }

        return $this->saveFile($images);
    }

    /**
     * обработка и сохранение размеров
     *
     * @param int $width
     * @param int $height
     * @param string $path
     */
    public function resizeImage(int $width, int $height, string $path)
    {
        if (ConfigManager::getValue('allow_watermark') && $this->extension != 'gif') {
            $this->generateWatermark();
        }

        $this->image->resize($width, $height, function ($constraint) {
            $constraint->aspectRatio();
            $constraint->upsize();
        })->save($path, ConfigManager::getValue('thumbnail_quality'), $this->extension);

    }

    private function generateWatermark()
    {
        $this->image->text(ConfigManager::getValue('watermark_text'), $this->image->width() / 2, $this->image->height() / 2, function ($font) {
            $font->file($this->fontWatermark);
            $font->size($this->fontOptions['size']);
            $font->color($this->fontOptions['color']['light']);
            $font->align('center');
            $font->valign('top');
            $font->angle($this->fontOptions['angle']);
        });
    }


    private function setDriver(): void
    {
        if (ConfigManager::getValue('image_library') != 2) {
            if (extension_loaded('imagick') && class_exists('Imagick'))	{
                $this->driver = 'imagick';

                if (!\Imagick::queryFormats('WEBP') && function_exists('imagewebp') && ConfigManager::getValue('image_library') != 1 ) {
                    $this->driver = 'gd';
                }
            }
        }

        $this->manager = new ImageManager(['driver' => $this->driver]);
    }

}