<?php

namespace Mnv\Core\Uploads;

use getID3;
use Mnv\Core\Singleton\SingletonTrait;

/**
 * Class ImageSizes
 * @package Mnv\Core\Uploads
 */
final class ImageSizes
{

    use SingletonTrait;

    /** @var array  */
    private static array $sizes = ['large', 'medium', 'small'];

    /**
     * Get image sizes and related file information.
     *
     * @param $image
     * @param $file
     * @return array
     */
    public static function get($image, $file): array
    {
        $imageSizes = $image ?? [];

        if (empty($file)) {
            return $imageSizes;
        }

        $filePath = GLOBAL_ROOT . $file['path'];
        $fileName = $file['fileName'];
        $fileFullPath = $filePath . $fileName;

        $imageSizes['fileFormat'] = self::fileExtension($fileName);
        $imageSizes['fileName'] = $fileName;

        if (file_exists($fileFullPath)) {
            $imageSizes['original'] = $file['path'] . $fileName;
        }

        foreach (self::$sizes as $size) {
            $sizePath = $filePath . $size . '/' . $fileName;
            if (file_exists($sizePath)) {
                $imageSizes[$size] = $file['path'] . $size . '/' . $fileName;
            }
        }

        // Handle SVG files separately
        if ($imageSizes['fileFormat'] === 'svg' && file_exists($fileFullPath)) {
            foreach (self::$sizes as $size) {
                $imageSizes[$size] = $file['path'] . $fileName;
            }
        }

        // Handle audio files (e.g., MP3)
        if (isset($file['mimeType']) && $file['mimeType'] === 'audio/mpeg') {
            $getID3 = new getID3();
            $fileInfo = $getID3->analyze($fileFullPath);
            $imageSizes['playtime_string'] = $fileInfo['playtime_string'] ?? null;
        }

        return $imageSizes;
    }

    /**
     * Get the file extension from a file name.
     *
     * @param string $name
     * @return string
     */
    public static function fileExtension(string $name): string
    {
        return pathinfo($name, PATHINFO_EXTENSION);
    }

}