<?php

namespace Mnv\Core\Uploads;

use Mnv\Core\ConfigManager;
use Mnv\Core\Filesystem\Filesystem;
use Mnv\Core\Locale\I18N;
use Symfony\Component\HttpFoundation\File\Exception\FileException;
use Symfony\Component\HttpFoundation\File\UploadedFile as SymfonyUploadedFile;
use Symfony\Component\HttpFoundation\Response as ResponseAlias;

/**
 * Class Uploader
 * @package Mnv\Core\Uploads
 */
class Uploader extends Filesystem
{

    const IMAGE_TYPE_GIF = 1;
    const IMAGE_TYPE_JPEG = 2;
    const IMAGE_TYPE_PNG = 3;
    const IMAGE_TYPE_WEBP = 4;

    /** @var array  */
    protected array $config = [];

    /** @var array  */
    protected array $options = [];

    /** @var int  */
    public int $response_status = ResponseAlias::HTTP_OK;

    /** @var SymfonyUploadedFile|mixed  */
    protected SymfonyUploadedFile $file;
    protected $uuid;

    /** @var ImageGenerator|ImagineGenerator */
    protected $generator;

    /** @var array  */
    private array $uploaded_file = [];

    /**
     * @param array $options
     * @param string $realPath
     * @param string $path
     * @param string|null $uuid
     * @param $managerId
     */
    public function __construct(array $options, string $realPath, string $path, ?string $uuid, $managerId)
    {
        $this->initializeOptions($options);
        $this->realPath = $realPath;
        $this->path     = $path;
        $this->uuid     = $uuid;

        $this->generator = $this->initializeGenerator($managerId);

        // TODO: доработать `request->file()`
//        if (request()->hasFile('file')) {
            $this->file = request()->files->get('file', '');

//            print_r($this->file);

            $this->populateUploadedFileData();
//        }
    }

    private function initializeOptions(array $options): void
    {
        $this->options = include __DIR__ . '/config/config.inc.php';
        if (!empty($options)) {
            $this->options = $options + $this->options;
        }
        $this->options['max_file_size'] = ConfigManager::getValue('max_file_size') * 1024 * 1024;
        $this->options['max_up_size'] = ConfigManager::getValue('max_up_size') * 1024 * 1024;
    }

    private function initializeGenerator(string $managerId)
    {
        if (ConfigManager::getValue('image_generation') === 'imagine') {
            return GeneratorFactory::imagineGenerator($this->realPath, $this->path, $this->uuid, $managerId);
        }

        return GeneratorFactory::imageGenerator($this->realPath, $this->path, $this->uuid, $managerId);
    }

    private function populateUploadedFileData(): void
    {
        $this->uploaded_file = [
            'name'      => $this->file->getClientOriginalName(),
            'tmp_name'  => $this->file->getPathname(),
            'mimeType'  => $this->file->getClientMimeType(),
            'size'      => $this->file->getSize(),
            'extension' => $this->file->getClientOriginalExtension(),
            'error'     => $this->file->getError(),
        ];
    }

    /**
     * Валидация загружаемого файла
     * @return bool
     */
    public function validate(): bool
    {
        if (empty($this->file)) {
            $this->response_status = ResponseAlias::HTTP_UNPROCESSABLE_ENTITY;
            return $this->setErrorResponse(ResponseAlias::HTTP_UNPROCESSABLE_ENTITY,'no_file_was_uploaded', 'fileManager:errors:27');
        }

        if ($this->file->getError()) {
            $this->response_status = ResponseAlias::HTTP_UNPROCESSABLE_ENTITY;
            return $this->setErrorResponse(ResponseAlias::HTTP_UNPROCESSABLE_ENTITY, $this->file->getError(), 'fileManager:errors:28');
        }

        if ($this->file->getSize() > SymfonyUploadedFile::getMaxFilesize()) {
            $this->response_status = ResponseAlias::HTTP_REQUEST_ENTITY_TOO_LARGE;
            return $this->setErrorResponse(ResponseAlias::HTTP_REQUEST_ENTITY_TOO_LARGE,'Request Entity Too Large', 'fileManager:errors:28');
        }

        if (!$this->file->getSize()) {
            $this->response_status = ResponseAlias::HTTP_UNPROCESSABLE_ENTITY;
            return $this->setErrorResponse(ResponseAlias::HTTP_UNPROCESSABLE_ENTITY,'file_size_empty', 'fileManager:errors:13');
        }

        if ($this->options['max_file_size'] && $this->file->getSize() > $this->options['max_file_size']) {
            $this->response_status = ResponseAlias::HTTP_BAD_REQUEST;
            return $this->setErrorResponse(ResponseAlias::HTTP_BAD_REQUEST,'max_file_size', 'fileManager:errors:26');
        }

        if ($this->options['min_file_size'] && $this->file->getSize() < $this->options['min_file_size']) {
            $this->response_status = ResponseAlias::HTTP_BAD_REQUEST;
            return $this->setErrorResponse(ResponseAlias::HTTP_BAD_REQUEST,'min_file_size', 'fileManager:errors:25');
        }

        if ($this->is_valid_image_file()) {
            if (!$this->validateImageSize()) {
                return false;
            }
        }

        if ($this->path === '') {
            $this->response_status = ResponseAlias::HTTP_BAD_REQUEST;
            return $this->setErrorResponse(ResponseAlias::HTTP_BAD_REQUEST,'root_path_error', 'fileManager:errors:18');
        }

        $this->ensureDirectoryExists($this->realPath);

        if ($this->exists($this->realPath . $this->uploaded_file['name'])) {
            $this->response_status = ResponseAlias::HTTP_BAD_REQUEST;
            return $this->setErrorResponse(ResponseAlias::HTTP_BAD_REQUEST,'duplicate_file', 'fileManager:errors:16');
        }

        return true;
    }

    public function upload(): bool
    {
        try {
            if (!preg_match($this->options['accept_file_types'], $this->uploaded_file['name'])) {
                $this->file->move($this->realPath, $this->uploaded_file['name']);
                $uploadedId = $this->generator->saveFile($this->uploaded_file);
            } else {
                try {
                    $uploadedId = $this->generator->init($this->file)->save();
                } catch (Exceptions\InvalidParamException $e) {
                    $this->response_status = ResponseAlias::HTTP_BAD_REQUEST;
                    $this->response = array('status' => ResponseAlias::HTTP_BAD_REQUEST, 'message' => I18N::locale("Невозможно загрузить файл на сервер", "Rasm qo'shildi!", "Image added!"), 'error' => 'unable_upload_file', 'type' => 'error');
                    return false;
                }
            }

            if (!$uploadedId) {
                $this->response_status = ResponseAlias::HTTP_BAD_REQUEST;
                $this->response = array('status' => ResponseAlias::HTTP_BAD_REQUEST, 'message' => I18N::locale("Невозможно загрузить файл на сервер", "Rasm qo'shildi!", "Image added!"), 'error' => 'unable_upload_file', 'type' => 'error');
                return false;
            }

            $this->response_status = ResponseAlias::HTTP_OK;
            $this->response = [
                'status' => ResponseAlias::HTTP_OK,
                'message' => I18N::locale("Файл загружен.", "Rasm qo'shildi!", "Image added!"),
                'error' => '',
                'fileId' => $uploadedId,
                'url' => $this->getFileUrl($uploadedId),
                'link' => '',
                'type' => 'success'
            ];

            return true;
        } catch (FileException $e) {
            error_log($e->getMessage());
            $this->response_status = ResponseAlias::HTTP_BAD_REQUEST;
            return $this->setErrorResponse(ResponseAlias::HTTP_BAD_REQUEST,'unable_upload_file', $e->getMessage());
        }
    }

    /**
     * Метод для получения картинки при загрузках в контент (tinymce)
     *
     * @param $fileId
     * @return string|null
     */
    public function getFileUrl($fileId): ?string
    {
        $file = connect('files')->where('fileId', $fileId)->get('array');
        if ($file) {
            $image = ImageSizes::init()->get(null, $file);
            return $image['original'] ?? '';
        }

        return null;
    }


    private function setErrorResponse($status, string $errorType, string $errorMessageKey): bool
    {
        $this->response = [
            'status'    => $status,
            'message'   => lang($errorMessageKey),
            'error'     => $errorType,
            'type'      => 'error'
        ];

        return false;
    }

    private function validateImageSize(): bool
    {
        list($img_width, $img_height) = $this->get_image_size();

        if ($this->options['max_up_size'] && $this->file->getSize() > $this->options['max_up_size']) {
            $this->response_status = ResponseAlias::HTTP_BAD_REQUEST;
            return $this->setErrorResponse(ResponseAlias::HTTP_BAD_REQUEST,'max_up_size', 'fileManager:errors:15');
        }

        if (($this->options['max_width'] && $img_width > $this->options['max_width']) || ($this->options['max_height'] && $img_height > $this->options['max_height'])) {
            $this->response_status = ResponseAlias::HTTP_REQUEST_ENTITY_TOO_LARGE;
            return $this->setErrorResponse(ResponseAlias::HTTP_REQUEST_ENTITY_TOO_LARGE,'max_width_height', 'fileManager:errors:21');
        }

        if (($this->options['min_width'] && $img_width < $this->options['min_width']) || ($this->options['min_height'] && $img_height < $this->options['min_height'])) {
            $this->response_status = ResponseAlias::HTTP_BAD_REQUEST;
            return $this->setErrorResponse(ResponseAlias::HTTP_BAD_REQUEST,'min_width_height', 'fileManager:errors:23');
        }

        return true;
    }


    /**
     * Валидация изображения
     * @return bool
     */
    protected function is_valid_image_file(): bool
    {
        return preg_match($this->options['accept_file_types'], $this->file->getClientOriginalName()) && $this->image_type() !== false;
    }

    protected function image_type()
    {
        $data = file_get_contents($this->file->getPathname(), false, null, 0, 4);

        if ($data === 'GIF8') {
            return self::IMAGE_TYPE_GIF;
        }

        if (bin2hex(substr($data, 0, 3)) === 'ffd8ff') {
            return self::IMAGE_TYPE_JPEG;
        }

        if (bin2hex(@$data[0]) . substr($data, 1, 4) === '89PNG') {
            return self::IMAGE_TYPE_PNG;
        }

        if ($data === 'RIFF') {
            return self::IMAGE_TYPE_WEBP;
        }

        return false;
    }

    private function get_image_size_with_library()
    {
        if (extension_loaded('imagick')) {
            try {
                $image = new \Imagick();
                if ($image->pingImage($this->file->getPathname())) {
                    $dimensions = [$image->getImageWidth(), $image->getImageHeight()];
                    $image->destroy();
                    return $dimensions;
                }
            } catch (\Exception $e) {
                error_log($e->getMessage());
            }
        }

        if ($this->options['image_library'] === 2) {
            $cmd = $this->options['identify_bin'] . ' -ping ' . escapeshellarg($this->file->getPathname());
            exec($cmd, $output, $error);
            if (!$error && !empty($output)) {
                $infos = preg_split('/\s+/', substr($output[0], strlen($this->file->getPathname())));
                return preg_split('/x/', $infos[2]);
            }
        }

        return false;
    }

    protected function get_image_size()
    {
        if ($this->options['image_library']) {
            return $this->get_image_size_with_library();
        }

        return function_exists('getimagesize') ? @getimagesize($this->file->getPathname()) : false;
    }

}