<?php

namespace  Mnv\Core\YaMetrika;

use GuzzleHttp\Client as GuzzleClient;
use GuzzleHttp\ClientInterface;
use Mnv\Core\Test\Log;
use Mnv\Core\YaMetrika\Exceptions\ClientException;
use Psr\Http\Client\ClientExceptionInterface;
use Psr\Http\Message\RequestInterface;

class Client
{
    public const API_BASE_PATH = 'https://api-metrika.yandex.ru';

    /**
     * @var array<string, mixed>
     */
    private array $config;

    private ClientInterface $httpClient;

    private Log $logger;

    /**
     * @param array<string, mixed> $config
     * @throws ClientException
     */
    public function __construct(array $config = [])
    {
        $this->config = array_merge([
            'token' => null,
        ], $config);

        if (empty($this->config['token'])) {
            throw new ClientException('Token is required');
        }

        $this->logger = $this->createDefaultLogger();

        $this->httpClient = $this->createDefaultHttpClient();
    }

    /**
     * Set AUTH token
     *
     * @param string $token
     * @return void
     */
    public function setToken(string $token): void
    {
        $this->config['token'] = $token;
    }

    /**
     * Set logger
     *
     * @param Log $logger
     * @return void
     */
    public function setLogger(Log $logger): void
    {
        $this->logger = $logger;
    }

    /**
     * Get logger
     *
     * @return Log
     */
    public function getLogger(): Log
    {
        return $this->logger;
    }

    /**
     * Set HTTP client
     *
     * @param ClientInterface $httpClient
     * @return void
     */
    public function setHttpClient(ClientInterface $httpClient): void
    {
        $this->httpClient = $httpClient;
    }

    /**
     * Get HTTP client
     *
     * @return ClientInterface
     */
    public function getHttpClient(): ClientInterface
    {
        return $this->httpClient;
    }

    /**
     * Execute HTTP request
     *
     * @param RequestInterface $request
     * @return array
     * @throws ClientException
     */
    public function execute(RequestInterface $request): array
    {
        try {
            $request = $request->withHeader('Authorization', sprintf('OAuth %s', $this->config['token']));
            $response = $this->httpClient->send($request);
        } catch (ClientExceptionInterface $e) {
            $this->logger->write($e->getMessage());

            throw new ClientException($e->getMessage(), $e->getCode(), $e);
        }

        return json_decode($response->getBody(), true);
    }

    /**
     * Create default logger
     *
     */
    protected function createDefaultLogger(): Log
    {
        return new Log('yandex.log');
    }

    /**
     * Create default HTTP client
     *
     * @return ClientInterface
     */
    protected function createDefaultHttpClient(): ClientInterface
    {
        return new GuzzleClient([
            'base_uri' => self::API_BASE_PATH,
        ]);
    }
}
