<?php

namespace Mnv\Models;

use Mnv\Core\Model;
use Mnv\Core\SortTable;
use Mnv\Core\Uploads\ImageSizes;
use Mnv\Http\Request;
use Mnv\Models\Exceptions\NoContentException;
use Mnv\Models\Exceptions\NotFoundException;

class User extends Model
{

    /** @var string */
    protected string $table = 'users';

    /** @var string */
    protected string $table_image = 'user_images';

    /** @var string */
    protected string $primaryKey = 'userId';

    /** @var string */
    protected string $orderBy = 'userId ASC';

    /** @var array|null */
    public ?array $oldUser = null;

    public function __construct(Request $request)
    {
        $this->id           = $request->get('id');
        $this->data         = $request->get('user');

        $this->sortBy       = $request->get('sortBy');
        $this->sortOrder    = $request->get('sortOrder');
    }

    protected function sorting()
    {
        if (!empty($this->filter['query'])) {
            connect()->whereAny(['loginName', 'fullName', 'firstName', 'lastName', 'phone'], 'like', "%" . $this->filter['query'] . "%");
        }
        if (!empty($this->filter['status'])) connect()->where('status', $this->filter['status']);


        if (!empty($this->filter['userType'])) {
            connect()->where('userType', $this->filter['userType']);
        }
    }

    /**
     * Получение всех пользователей
     *
     * @param int $limit
     * @param int $page
     */
    public function all( int $limit, int $page)
    {
        /** Фильтрация / сортировка */
        $this->sorting();

       return parent::all($limit, $page);
    }

    /**
     * Получение кол-во пользователей
     */
    public function total(): void
    {
        /** Фильтрация / сортировка */
        $this->sorting();
        parent::total();
    }


    /**
     * Одобрение пользователя
     *
     * @return bool
     *
     * @throws NoContentException
     * @throws NotFoundException
     */
    public function approve() : bool
    {
        if (empty($this->id)) {
            throw new NotFoundException("Not Found: ID is required.");
        }

        $currentRecord = connect($this->table)->where($this->primaryKey, $this->id)->get('array');

        // Throw exception if no record is found for the given `id`
        if (!$currentRecord) {
            throw new NoContentException("No Content: Record not found for the given ID.");
        }

        if (connect($this->table)->where($this->primaryKey, $this->id)->update(['status' => 1])) {
            return true;
        }

        return false;
    }


    public function prepare(array $data, int $managerId): bool
    {

        $data['modifiedBy']   = $managerId;
        $data['modifiedOn']   = gmdate('Y-m-d H:i:s');

        if (empty($this->id)) {

            $data['password']      = \password_hash($data['password'], \PASSWORD_DEFAULT);
            $data['addedBy']       = $managerId;
            $data['addedOn']       = gmdate('Y-m-d H:i:s');
            $data['registered']    = \time();
            $data['last_login']    = \time();
            $data['verified']      = 1;

            if ($this->id = $this->insert($data)) {
                return true;
            }
        } else {

            if (!empty($data['newPassword'])) {
                $data['password'] = \password_hash($data['newPassword'], \PASSWORD_DEFAULT);
            }
            unset($data['newPassword']);

            if ($this->update($data)) {
                return true;
            }
        }

        return false;
    }


    /**
     * Проверяем существует ли нововведений логин в базе данных
     * @param $loginName
     */
    /**
     * @param $loginName
     * @return mixed|string|null
     */
    public function checkExistenceLoginName($loginName): ?string
    {
        if (!empty($this->id)) connect()->where($this->primaryKey,'<>', $this->id);
        if ($fileName = connect($this->table)->select('loginName')->where('LOWER(loginName)', $loginName)->getValue())
            return $fileName;

        return null;
    }
    /**
     * Проверяем существует ли нововведений логин в базе данных
     *
     * @param $manager
     * @return bool
     */
    public function existUserLoginName($manager) : bool
    {
        if (connect($this->table)->where('loginName = ? AND userId <> ?', [$manager->loginName, $manager->userId])->get())  return true;

        return false;
    }


    /**
     * Удаление фото
     *
     * @return bool
     */
    /**
     * @param int $userId
     * @return bool
     */
    public function removeUserImage(int $userId) : bool
    {
        if (empty($userId)) {
            return false;
        }

        connect($this->table_image)->where($this->primaryKey,  $userId)->delete();

        return true;
    }

    /**
     * Проверка на совпадение loginName
     * @param string $loginName
     * @return string|null
     */
    public function checkLogin(string $loginName): ?string
    {
        if (!empty($this->id)) connect()->where($this->primaryKey,'<>', $this->id);
        if ($loginName = connect($this->table)->select('loginName')->where('loginName', $loginName)->getValue()) {
            return $loginName;
        }

        return null;
    }

    /**
     * Проверка на совпадение email
     * @param string $email
     * @return string|null
     */
    public function checkEmailUser(string $email): ?string
    {
        if (!empty($this->id)) connect()->where($this->primaryKey,'<>', $this->id);
        if ($email = connect($this->table)->select('email')->where('email', $email)->getValue()) {
            return $email;
        }

        return null;
    }



    /**
     * Проверка существует ли такой менеджер или да то возвращаем его
     *
     * @param int $userId
     */
    public function getOldDataUser(int $userId) : void
    {
        $this->oldUser = connect($this->table)->where($this->primaryKey, $userId)->get('array');
    }



    /**
     * Проверяем существует ли другой пользователь с указанными правами доступа администратор `[Role::DEVELOPER, Role::ADMIN]`
     * если он один такой то запрет на изменение
     * всегда должен существовать один пользователь с правами `[Role::DEVELOPER, Role::ADMIN]`
     *
     * @param array $manager
     * @param string $userType
     * @return bool
     */
    public function groupByRole(array $manager, string $userType) : bool
    {
        if (!connect($this->table)->where('userId <> ? AND userType = ?', [$manager['userId'], $userType])->get())  return true;

        return false;
    }


}
