<?php


namespace Mnv\Modules\Cars;


use Mnv\Core\Uploads\ImageCollection;

/**
 * Class Model
 * @package Mnv\Modules\Cars
 */
class Model
{
    /** @var int */
    private $modelId;
    /** @var array  */
    public $model = [];

    /** @var string $type E = equipments / M = modifications */
    private $type;

    private $isEquipments;
    private $isOptions;

    public function __construct($modelId, $type = 'E', $isEquipments = false, $isOptions = false)
    {
        $this->modelId = $modelId;
        $this->type = $type;
        $this->isEquipments = $isEquipments;
        $this->isOptions = $isOptions;
    }

    public static function getMinPriceEquipments($modelId)
    {
        return connect('car_equipments')->select('price')->where('status', 'V')->where('modelId', $modelId)->min('price', 'min_price')->getValue();
    }

    public function getModel(): array
    {
        $this->model['images'] = $this->getFiles('car_model_images', 'modelId', $this->modelId);
        // получить цвета из модели
        $this->model['exterior_colors'] = $this->getModelBColors($this->modelId);
        $this->model['interior_colors'] = $this->getModelIColors($this->modelId);

        if ($this->type == 'E') {
            $this->model['equipments'] = $this->getModelEquipments();
            $exterior_colors = [];
            $interior_colors = [];
            // группируем полученные цвета из комплектации
            foreach ($this->model['equipments'] as $equipment) {
                if (isset($equipment['exterior_colors'])) {
                    foreach ($equipment['exterior_colors'] as $exterior_color) {
                        if (isset($exterior_colors[$exterior_color['code']]) && $exterior_colors[$exterior_color['code']]['code'] == $exterior_color['code']) {
                            continue;
                        }
                        $exterior_colors[$exterior_color['code']] = $exterior_color;
                    }
                }
                if (isset($equipment['interior_colors'])) {
                    foreach ($equipment['interior_colors'] as $interior_color) {
                        if (isset($interior_colors[$interior_color['code']]) && $interior_colors[$interior_color['code']]['code'] == $interior_color['code']) {
                            continue;
                        }
                        $interior_colors[$interior_color['code']] = $interior_color;
                    }
                }
            }

//            print_r($this->model);
        }
        else if  ($this->type == 'M') {
            $this->model['modifications'] = $this->getModifications();
        }

        return $this->model;
    }

    public function getModifications(): array
    {
        $modifications = connect('car_modifications')->select('name, power, power_reserve, modelId, modificationId')->where('status', 'V')->where('modelId', $this->modelId)->orderBy('orderBy ASC')->getAll('array');
        return collect($modifications)->map(function ($modification) {
            if ($this->isEquipments) {
                $modification['equipments'] = $this->getModificationEquipments($modification['modificationId']);
            }
            $modification['specifications'] = $this->getSpecifications($modification['modificationId']);

            return $modification;
        })->all();

    }
    public function getModificationEquipments($modificationId)
    {
        if (!empty($modificationId)) {
            $equipments = connect('car_equipments')->select('equipmentId, title AS name, summary')->where('status', 'V')->where('modelId', $this->modelId)->where('modificationId', $modificationId)->orderBy('publishedOn ASC')->getAll('array');
            return collect($equipments)->map(function ($item) {
                $item['options'] = $this->getOptions($item['equipmentId']);
                $item['exterior_colors'] = $this->getEquipmentBColors($item['equipmentId']);
                $item['interior_colors'] = $this->getEquipmentIColors($item['equipmentId']);

                return $item;
            })->all();
        }
    }

    /**
     * Получить комплектации по модели
     * @return array
     */
    public function getModelEquipments(): array
    {
        $equipments = connect('car_equipments')->select('title, price, oldPrice, equipmentId, modificationId, summary')->where('status', 'V')->where('modelId', $this->modelId)->orderBy('publishedOn ASC')->getAll('array');
        return collect($equipments)->map(function ($item) {
            $item['images'] = $this->getFiles('car_equipment_images', 'equipmentId', $item['equipmentId']);
            $item['modification']   = $this->getModification($item['modificationId']);
            if ($this->isOptions) $item['options'] = $this->getOptions($item['equipmentId']);
//            $item['exterior_colors'] = $this->getEquipmentBColors($item['equipmentId']);
//            $item['interior_colors'] = $this->getEquipmentIColors($item['equipmentId']);


            return $item;
        })->all();
    }

    public static function getModification($modificationId): ?array
    {
        return connect('car_modifications')->select('modificationId, name, power, power_reserve, engine, acceleration, transmissions, gear')->where('modificationId', $modificationId)->get('array');

    }

    /**
     * Спецификация
     * @param $modificationId
     * @return array|null
     */
    public function getSpecifications($modificationId): ?array
    {
        $specifications = connect('car_specifications')->select('specificationId AS ID, name')->orderBy('orderBy')->whereNull('parentId',  false)->getAll('array');

        return collect($specifications)->map(function ($specification) use ($modificationId) {
            $properties = connect('car_specifications')->select('specificationId AS ID, parentId, name')->where('parentId', $specification['ID'])->whereNull('parentId',  true)->orderBy('orderBy ASC')->getAll('array');

            $values = [];
            collect($properties)->map(function ($item) use (&$values, $modificationId) {
                $property = connect()->table('car_specifications_value')->select('value, unit')->where('specificationId', $item['ID'])->where('modificationId', $modificationId)->get('array');
                if (!empty($property['value'])) {
                    $values[$item['name']] = $property['value'];
                }
            })->all();

            $specification['properties'] = $values;

            return $specification;
        })->all();
    }

    /**
     * Опции комплектации
     */
    private function getOptions($equipmentId): array
    {
        $values = [];
        $options = connect('car_options')->orderBy('orderBy ASC')->whereNull('parentId',  false)->indexKey('name')->getAllIndexes('array');
        if (!empty($options)) {
            collect($options)->map(function ($option) use (&$values, $equipmentId) {
                $values[$option['name']] = connect('car_options_value')->usingJoin('car_options AS ceo', 'optionId')->where('ceo.parentId', $option['optionId'])->where('equipmentId', $equipmentId)->whereNull('ceo.parentId', true)->indexKey('optionId')->valueKey('name')->getAllIndexes('array');
            })->all();
        }
        return $values;
    }

    /**
     * Цвет кузова
     * @param $modelId
     * @return array|false|mixed|string|null
     */
    private function getModelBColors($modelId): ?array
    {
        $colors = connect()->table('car_model_colors AS cec')->usingJoin('car_colors AS cc', 'colorId')->select('cc.colorName AS name, cc.colorCode AS code, cec.fileId, cec.price, cec.menu')->where('cec.type', 'bcolors')->where('cec.modelId', $modelId)->orderBy('cec.orderBy ASC')->getAll('array');
        return collect($colors)->map(function ($color) {
            if ($file = getFileInfo($color['fileId'])) {
                $color['image'] = ImageCollection::init()->get(null, $file);
            }

            return $color;
        })->all();
    }

    /**
     * Цвет салона
     * @param $modelId
     * @return array|false|mixed|string|null
     */
    private function getModelIColors($modelId): ?array
    {
        $colors = connect()->table('car_model_colors AS cec')->usingJoin('car_colors AS cc', 'colorId')->select('cc.colorName AS name, cc.colorCode AS code, cec.fileId, cec.price')->where('cec.type', 'icolors')->where('cec.modelId', $modelId)->orderBy('cec.orderBy ASC')->getAll('array');
        return collect($colors)->map(function ($color) {
            if ($file = getFileInfo($color['fileId'])) {
                $color['image'] = ImageCollection::init()->get(null, $file);
            }

            return $color;
        })->all();

    }

    public function getEquipmentBColors($equipmentId)
    {

        $images = connect()->table('car_equipment_colors AS cec')->usingJoin('car_colors AS cc', 'colorId')->select('cc.colorName AS name, cc.colorCode AS code, cec.fileId, cec.price')->where('cec.type', 'bcolors')->where('cec.equipmentId', $equipmentId)->orderBy('cec.orderBy ASC')->getAll('array');
        return collect($images)->map(function ($item)  {
            if ($file = $this->getFileInfo($item['fileId'])) {
                $item = ImageCollection::init()->get($item, $file);
            }
            return $item;
        })->all();
    }

    public function getEquipmentIColors($equipmentId)
    {
        $images = connect()->table('car_equipment_colors AS cec')->usingJoin('car_colors AS cc', 'colorId')->select('cc.colorName AS name, cc.colorCode AS code, cec.fileId, cec.price')->where('cec.type', 'icolors')->where('cec.equipmentId', $equipmentId)->orderBy('cec.orderBy ASC')->getAll('array');
        return collect($images)->map(function ($item) {
            if ($file = $this->getFileInfo($item['fileId'])) {
                $item = ImageCollection::init()->get($item, $file);
            }
            return $item;
        })->all();
    }


    /** РАБОТА С ФАЙЛАМИ / ИЗОБРАЖЕНИЕМ */

    protected function getFileInfo($fileId): ?array
    {
        return connect('files')->where('fileId', $fileId)->get('array');
    }

    /** получить все из таблицы images модели  */
    public function getFiles($table, $primaryKey, $sectionId): ?array
    {
        $files = [];
        if ($images = connect($table)->where($primaryKey, $sectionId)->select('fileId, type, position, title, description, link')->orderBy('orderBy ASC')->getAll('array')) {
            collect($images)->map(function ($item) use (&$files) {
                if ($file = $this->getFileInfo($item['fileId'])) {
                    if ($item['type'] == 'general') {
                        $files['general'] = ImageCollection::init()->get($item, $file);
                    } elseif ($item['type'] == 'gallery') {
                        $files['gallery'][] = ImageCollection::init()->get($item, $file);
                    }
                    else {
                        $files['docs'][] = ImageCollection::init()->get($item, $file);
                    }
                }

                return $files;
            })->toArray();
        }

        return $files;
    }

}